-- File: media/lua/client/DST/Contrib/Tailoring.lua
-- Purpose: Contribute Tailoring-specific lines to Detailed Skill Tooltips (DST)
-- Build: 42.x
-- Inputs (via DST):
--   ctx.level (number 0..10) – Tailoring level for the preview
--   ctx.add(string)          – Append a formatted line to the tooltip
-- Outputs: none (side-effect: adds lines to tooltip)

require "API/DST_API"
require "API/DST_Helpers"

local DST = rawget(_G, "DST") or {}
local ST  = DST and DST.SkillTooltips
if not ST then return end

local KEY = "Tailoring"

-- Contributor -----------------------------------------------------------------
-- Shows fabric padding values at current level, patch recovery chance, and the
-- L8 "repair holes fully" milestone.
ST.addContributor(KEY, function(ctx)
    local lvl = ctx.getLevel()

    -- Defense tables per fabric (Scratch / Bite / Bullet) by Tailoring level.
    -- Source: PZwiki (B42 page). ⚠ Keep in sync if upstream changes. 
    -- Rags: Scratch only
    local RAG_S = {1,1,1,1,2,2,3,3,4,4,5}

    -- Denim: S/B/Bu
    local DEN_S = {1,1,1,1,2,2,3,3,4,4,5}
    local DEN_B = {1,1,2,3,4,5,6,7,8,9,10}
    local DEN_U = {1,1,1,1,2,2,3,3,4,4,5}

    -- Leather: S/B/Bu
    local LEA_S = {1,1,2,3,4,5,6,7,8,9,10}
    local LEA_B = {2,2,4,6,8,10,12,14,16,18,20}
    local LEA_U = {1,1,2,3,4,5,6,7,8,9,10}

    -- Patch recovery chance: 10% + 5% * lvl, capped at 60% (L10)
    local recoverPct = math.min(10 + 5 * lvl, 60)

    -- Header
    ctx.addHeader(ST.getText("IGUI_DST_Tailoring_hdr_Defense"))

    -- Fabric lines
    ctx.add(ST.getText("IGUI_DST_Tailoring_val_Rag",
        tostring(RAG_S[lvl])))

    ctx.add(ST.getText("IGUI_DST_Tailoring_val_Denim",
        tostring(DEN_S[lvl]), tostring(DEN_B[lvl]), tostring(DEN_U[lvl])))

    ctx.add(ST.getText("IGUI_DST_Tailoring_val_Leather",
        tostring(LEA_S[lvl]), tostring(LEA_B[lvl]), tostring(LEA_U[lvl])))

    -- Recovery chance
    ctx.addHeader(ST.getText("IGUI_DST_Tailoring_hdr_Recovery"))
    ctx.add(ST.getText("IGUI_DST_Tailoring_val_Recovery", tostring(recoverPct)))

    -- L8+: fully repair holes with matching fabric
    if lvl >= 8 then
        ctx.addSeparator()
        ctx.add(ST.getText("IGUI_DST_Tailoring_val_FixHoles"))
    end
end)

ST.addAutoLearnRecipes(KEY)
ST.addUnlockCraftRecipes(KEY)
ST.addUnlockBuildRecipes(KEY)
